# メソッド呼び出し(super・ブロック付き・yield) - Ruby 3.4 リファレンスマニュアル

## 目次
- [super](#super)
- [ブロック付きメソッド呼び出し](#ブロック付きメソッド呼び出し)
- [yield](#yield)
- [ブロックパラメータの挙動](#ブロックパラメータの挙動)
- [番号指定パラメータ](#番号指定パラメータ)
- [it](#it)
- [.() および ::() 形式のメソッド呼び出し](#call形式のメソッド呼び出し)

---

## メソッド呼び出しの基本

### 例
```ruby
foo.bar()
foo.bar
bar()
print "hello world\n"
print Class.new
Class::new
```

### 文法
```
[式 '.'] 識別子 ['(' [['*'] 式] ... ['&' 式] ')']
[式 '::'] 識別子 ['(' [['*'] 式] ... ['&' 式] ')']
```

メソッド呼び出し式はレシーバ（`.` の左側の式の値）のメソッドを呼び出します。レシーバが指定されないときは `self` のメソッドを呼び出します。

### . と :: の違い
`.` と `::` はほぼ同じ意味ですが、定数を表す場合は `::` を使う必要があります（例：`Math::PI`）。

定数として見なされるため、大文字で始まるメソッド名の場合は以下のいずれかを使用します：

```ruby
Klass.Foo      # . を使用
Klass::Foo()   # 括弧でメソッド呼び出しを明示
```

### メソッド名の規則
メソッド名には識別子の他、`?` または `!` を末尾に付けることができます：
- `?`：述語メソッド（真偽値を返す）
- `!`：破壊的な作用をするメソッド（例：`tr` と `tr!`）

### 引数の展開（スプラット演算子）
`*` がついている場合、引数が展開されます：

```ruby
foo(1,*[2,3,4])        # foo(1,2,3,4) と同じ
foo(1,*)               # foo(1) と同じ
foo(1,*[2,3,4],5)      # foo(1,2,3,4,5) と同じ
foo(1,*[2,3,4],5,*[6]) # foo(1,2,3,4,5,6) と同じ
```

### ブロックの渡し方
最後の引数に `&` がついている場合、手続きオブジェクト（`Proc`）またはメソッドオブジェクト（`Method`）がブロックとして渡されます。

### アクセス制限
- **private** メソッド：関数形式（レシーバ省略）でのみ呼び出し可能
- **protected** メソッド：そのメソッドを持つオブジェクトのメソッド定義式内でのみ呼び出し可能

### ハッシュ式の省略
メソッド呼び出しの末尾にハッシュを渡す場合、`{` と `}` を省略できます。これでキーワード引数を渡せます：

```ruby
method(key1: value1, key2: value2)
```

### セーフナビゲーション演算子（&.）
`&.` を使うことで、レシーバが `nil` の場合の安全な呼び出しが可能です：

```ruby
foo = 13
foo&.to_s      # => "13"
foo = nil
foo&.to_s      # => nil（エラーなし）
```

レシーバが `nil` の場合：
- 引数の評価が行われない
- メソッド呼び出しが行われない
- `nil` を返す

`&.` は要素代入にも使えます：

```ruby
foo&.bar = "abc"   # bar= メソッド向け
```

---

## super

### 文法
```
super
super(式, ... )
```

`super` は現在のメソッドがオーバーライドしているメソッドを呼び出します。

**括弧と引数が省略された場合**：現在のメソッドの引数がそのまま引き渡されます。

**引数を渡さない場合**：`super()` と括弧を明示します。

### 例
```ruby
class Foo
  def foo(arg=nil)
    p arg
  end
end

class Bar < Foo
  def foo(arg)
    super(5)       # 5 を引数にして呼び出す
    super(arg)     # arg を引数にして呼び出す
    super          # arg を引数にして呼び出す
    arg = 1
    super          # 1 を引数にして呼び出す
    super()        # 引数なしで呼び出す
  end
end

Bar.new.foo(5)
```

---

## ブロック付きメソッド呼び出し

### 例
```ruby
[1,2,3].each do |i|
  print i*2, "\n"
end

[1,2,3].each {|i| print i*2, "\n" }
```

### 文法
```
method(arg1, arg2, ...)  do [|式 ... |] 式 ... end
method(arg1, arg2, ...) '{' [|式 ... |] 式 ... '}'
method(arg1, arg2, ..., '&' proc_object)
```

ブロック付きメソッドは制御構造の抽象化に用いられます。`do ... end` または `{ ... }` で囲まれたコードの断片（**ブロック**）をメソッドの後ろに付けて呼び出します。

### 結合力の違い
`{ ... }` の方が `do ... end` より強く結合します：

```ruby
foobar a, b do body end   # foobar の引数は a, b の値とブロック
foobar a, b { body }      # ブロックはメソッド b の引数
```

### ブロック内のスコープ

ブロック内で初めて代入されたローカル変数はそのブロック内でのみ有効です：

```ruby
foobar {
  i = 20    # ローカル変数 i が宣言（ブロック内のみ有効）
}
print defined? i   # false

foobar a, b do
  i = 11          # 別の変数 i
end
```

ブロック外で定義された変数はブロック内から参照可能です：

```ruby
i = 10
[1,2,3].each do |m|
  p i * m   # i を参照可能
end
```

### Proc オブジェクトとしてのブロック渡し

ブロックを変数に保存して後から渡すことができます：

```ruby
pobj = proc {|v| p v }
[1,2,3].each(&pobj)
# => 1
#    2
#    3
```

`to_proc` メソッドを持つオブジェクトは `&` 修飾で渡せます：

```ruby
class Foo
  def to_proc
    Proc.new {|v| p v}
  end
end

[1,2,3].each(&Foo.new)
# => 1
#    2
#    3
```

### 戻り値
ブロック付きメソッドの戻り値は通常のメソッド同様ですが、`break` で中断された場合は `nil` を返します。`break` に引数を指定した場合はその値が戻り値になります。

---

## yield

### 文法
```
yield '(' [式 [',' 式 ... ]] ')'
yield [式 [',' 式 ... ]]
```

自分で定義したブロック付きメソッドでブロックを呼び出すときに使います。`yield` に渡された値はブロック記法の `|` と `|` の間のブロックパラメータに代入されます。

### 例

**基本的な使用法**：
```ruby
def foo
  yield(1,2)
end

foo {|a,b| p [a, b] }     # => [1, 2]
foo {|a, b| p a + b }     # => 3
```

**複数回の yield**：
```ruby
def bar
  yield 10
  yield 20
  yield 30
end

bar {|v| p v + 3 }
# => 13
#    23
#    33
```

**イテレータの実装例**：
```ruby
def iich(arr)
  idx = 0
  while idx < arr.size
    yield(arr[idx])
    idx += 1
  end
end

sum = 0
iich([1, 4, 9, 16, 25]) {|elem| sum += elem }
p sum   # => 55
```

### yield の挙動
- ブロック内で最後に評価した式の値を返します
- `next` で中断された場合は `nil` を返します
- `next` に引数を指定した場合はその値が戻り値になります
- ブロックが渡されていない場合は `LocalJumpError` 例外が発生します

---

## ブロックパラメータの挙動

メソッド呼び出しと異なり、lambda でないブロックを呼び出したときは以下が成り立ちます：

- 引数の数が違ってもエラーになりません
- 配列をひとつ渡したときに展開されることがあります

### 例
```ruby
def foo
  yield 1,2,3
end

foo{|v| p v}       # => 1

def bar
  yield [1,2,3]
end

bar{|a, b, c| p a} # => 1

def hoge
  yield [1,2,3],4,5
end

hoge{|a, b, c| p a} # => [1,2,3]
```

---

## 番号指定パラメータ

ブロックに渡された値を参照するには、暗黙に定義される `_1`、`_2` といった変数を用いる方法があります。番号指定パラメータは `_1` から `_9` までの9つが使用可能です。

### 例
```ruby
def foo
  yield "a", "b", "c"
end

foo{|a, b, c| p [a, b, c] }  # => ["a", "b", "c"]
foo{ p [_1, _2, _3] }        # => ["a", "b", "c"]
```

### 制限
ブロックパラメータと番号指定パラメータを同時に使うことはできません：

```ruby
foo {|a, b, c| p [_1, a] }   # => SyntaxError
```

### _1 の意味の変化
ブロック内で `_2` 以降が使用されているかどうかで、`_1` の意味が異なります：

```ruby
def foo
  yield ["a", "b", "c"]
end

foo {
  p _1   # => ["a", "b", "c"]
}

foo {
  p _1   # => "a"
  p _2   # => "b"
}
```

これはブロックパラメータの定義個数による代入ルールに対応します：

```ruby
foo {|a|
  p a   # => ["a", "b", "c"]
}

foo {|a,b|
  p a   # => "a"
  p b   # => "b"
}
```

---

## it

ブロックに渡された値は `it` という名前で参照することもできます。

### 例
```ruby
[1, 2, 3].map { it * 2 }   # => [2, 4, 6]

{a: 1, b: 2}.each { p it }
# => [:a, 1]
#    [:b, 2]
```

### it の特性
`it` は予約語ではないため、変数やメソッドの名前として使うこともできます：

```ruby
it = 0         # 警告されない

def it         # 警告されない
  # ...
end
```

### スコープと優先順位
ローカル変数 `it` が存在する場合、`it` はローカル変数の値を参照します：

```ruby
it = 0
[1, 2, 3].map { it * 2 }   # => [0, 0, 0]
```

### ネスト
`it` はネストすることができます：

```ruby
['foo', 'bar'].each { it.each_char { p it } }
# => "f"
#    "o"
#    "o"
#    "b"
#    "a"
#    "r"
```

### 制限
- 通常のブロックパラメータを定義した場合、そのブロックでは `it` は使用できません：
  ```ruby
  [1, 2, 3].map { |x| it * 2 }   # => SyntaxError
  ```

- 1つのブロックで番号指定パラメータと `it` の両方を使うことはできません：
  ```ruby
  [1, 2, 3].map { _1 + it }   # => SyntaxError
  ```

---

## .() および ::() 形式のメソッド呼び出し

### 説明
下記は `call` メソッドの糖衣構文です。`Proc` 以外のオブジェクトに対しても（`call` メソッドさえ定義されていれば）使えます。

### 例
```ruby
foo.(100)      # foo.call(100) と全く同じ意味
foo::(100)     # foo.call(100) と全く同じ意味
```

### 文法
```
式 '.' '(' [['*'] 式] ... ['&' 式] ')'
式 '::' '(' [['*'] 式] ... ['&' 式] ')'
```

---

## ライセンス
[![Creative Commons License](https://i.creativecommons.org/l/by/3.0/88x31.png)](https://creativecommons.org/licenses/by/3.0/)

- [フィードバックを送る](https://github.com/rurema/doctree/issues/new)
- [このマニュアルを編集する](https://github.com/rurema/doctree/edit/master/refm/doc/spec/call.rd#L1)
