# 制御構造 (Ruby 3.4 リファレンスマニュアル)

## 制御構造

Rubyの制御構造は式であり、値を返すものがあります。C言語やPerlから引き継いだ構造に加え、メソッド呼び出しやブロック付きメソッド呼び出しという機能があります。

---

## 条件分岐

### if

```ruby
if age >= 12 then
  print "adult fee\n"
else
  print "child fee\n"
end

gender = if foo.gender == "male" then
  "male"
else
  "female"
end
```

**文法:**

```
if 式 [then]
  式 ...
[elsif 式 [then]
  式 ... ]
...
[else
  式 ... ]
end
```

条件が真の場合、then以下の式を評価します。複数のelsif節を指定でき、全て偽の場合はelse節を実行します。返り値は条件が成立した節の最後の式、またはnilです。

Rubyではfalseとnilだけが偽で、0や空文字列を含めそれ以外は全て真です。

#### if 修飾子

```ruby
print "debug\n" if $DEBUG
```

**文法:** `式 if 式`

右辺の条件が成立する時に左辺を評価し、その結果を返します。条件が不成立ならnilを返します。

### unless

```ruby
unless baby?
  feed_meat
else
  feed_milk
end
```

**文法:**

```
unless 式 [then]
  式 ...
[else
  式 ... ]
end
```

ifと反対で、条件が偽の時にthen以下を評価します。elsif節は使用できません。

#### unless 修飾子

```ruby
print "stop\n" unless valid(passwd)
```

**文法:** `式 unless 式`

右辺の条件が不成立の時に左辺を評価します。条件成立ならnilを返します。

### case

```ruby
case $age
when 0 .. 2
  "baby"
when 3 .. 6
  "little child"
when 7 .. 12
  "child"
when 13 .. 18
  "youth"
else
  "adult"
end
```

**文法:**

```
case [式]
[when 式 [, 式] ...[, `*' 式] [then]
  式..]..
[when `*' 式 [then]
  式..]..
[else
  式..]
end
```

一つの式に対する一致判定で分岐します。when節の式を評価した結果をレシーバ、caseの式を引数として===演算子を呼び出し、最初に真を返したwhen節を実行します。

```ruby
ary = [1,2,3]

case v
when *ary
  # ..
end
```

上記は以下と等価です:

```ruby
case v
when 1, 2, 3
  # ..
end
```

caseの式を省略した場合、when条件が真の最初の式を評価します:

```ruby
foo = false
bar = true
quu = false

case
when foo then puts 'foo is true'
when bar then puts 'bar is true'
when quu then puts 'quu is true'
end
# "bar is true"と表示される
```

Ruby 2.7以降はcaseはinキーワードによるパターンマッチを提供しています:

```ruby
case {a: 1, b: 2, c: 3}
in a: Integer => m
  "matched: #{m}"
else
  "not matched"
end
# => "matched: 1"
```

---

## 繰り返し

### while

```ruby
ary = [0,2,4,8,16,32,64,128,256,512,1024]
i = 0
while i < ary.length
  print ary[i]
  i += 1
end
```

**文法:**

```
while 式 [do]
  ...
end
```

条件が真の間、本体を繰り返し実行します。nilを返します。break引数により戻り値を指定できます。

#### while 修飾子

```ruby
sleep(60) while io_not_ready?
```

**文法:** `式 while 式`

右辺が真の間、左辺を繰り返し実行します。

```ruby
begin
  res = get_response()
end while res == 'Continue'
```

begin節の場合は最初に一回評価してから繰り返します。

### until

```ruby
until f.eof?
  print f.gets
end
```

**文法:**

```
until 式 [do]
  ...
end
```

条件が真になるまで本体を繰り返し実行します。nilを返します。

#### until 修飾子

```ruby
print(f.gets) until f.eof?
```

**文法:** `式 until 式`

右辺が真になるまで左辺を繰り返し実行します。

```ruby
begin
  res = get_response()
end until res == 'OK'
```

### for

```ruby
for i in [1, 2, 3]
  print i*2, "\n"
end
```

**文法:**

```
for lhs ...  in 式1 [do]
  式2..
end
```

オブジェクトの各要素に対して本体を繰り返し実行します。以下と等価です:

```ruby
(式1).each { |lhs..| 式2.. }
```

forはローカル変数のスコープに影響を及ぼさない点が異なります。

複数ループ変数の例:

```ruby
for i,j in [[1,2], [3,4], [5,6]]
  p [i,j]
end
# => [1, 2]
#    [3, 4]
#    [5, 6]
```

配列要素を複数個ずつ取得しながらループすることはできません:

```ruby
for i,j in [1, 2, 3]
  p [i,j]
end
# => [1, nil]
#    [2, nil]
#    [3, nil]
```

代わりにカスタムイテレータを定義します:

```ruby
class Array
  def each2
    i = 0
    while i < self.size
      yield self[i], self[i+1]
      i += 2
    end
  end
end
```

### break

```ruby
i = 0
while i < 3
  print i, "\n"
  break
end
```

**文法:**

```
break
break val
```

最も内側のループ(while、until、for、イテレータ)を脱出します。C言語と異なり、caseを抜けません。

ループを抜けたforやイテレータはnilを返します。引数指定時はその値を返します。

### next

```ruby
# 空行を捨てる
ARGF.each_line do |line|
  next if line.strip.empty?
  print line
end
```

**文法:**

```
next
next val
```

最も内側のループの次の繰り返しにジャンプします。yieldから脱出します。

nilを返します。引数指定時はその値を返します。

### redo

**文法:**

```
redo
```

ループ条件をチェックせず、現在の繰り返しをやり直します。

#### イテレータにおけるbreak、next、redo

```ruby
def iter
  # (a)
  #  :
  # (b)
  yield
  # (c)
  #  :
  # (d)
end
iter { redo  }  # -> (b) へ飛ぶ
iter { next  }  # -> (c) へ飛ぶ
iter { break }  # -> (d) へ飛ぶ
```

詳細な挙動:

```ruby
def iter(var = p("(a)"))
  yield
  p "(c)"
ensure
  p "(d)"
end
iter { p "(b)"; redo  }     # -> (a) .. (b)(b)(b)(b) ...
iter { p "(b)"; next  }     # -> (a) .. (b)(c) .. (d)
iter { p "(b)"; break }     # -> (a)..(b)(d)
```

### retry

**文法:**

```
retry
```

rescue節でbegin式をはじめから実行します。

```ruby
begin
  do_something # exception raised
rescue
  # handles error
  retry  # restart from beginning
end
```

rescue節以外で使用するとSyntaxErrorが発生します。

---

## 例外処理

### raise

```ruby
raise "you lose"  # RuntimeError を発生させる
raise SyntaxError, "invalid syntax"
raise SyntaxError.new("invalid syntax")
raise             # 最後の例外を再発生
```

**文法:**

```
raise
raise messageまたはexception
raise error_type, message
raise error_type, message, traceback
```

例外を発生させます。第一形式では直前の例外を再発生、第二形式では文字列ならRuntimeError例外を発生、例外オブジェクトならそれを発生、第三形式では指定された例外をメッセージ付きで発生、第四形式は第三引数にスタック情報を指定します。

発生した例外はbegin式のrescue節で捕捉できます。`rescue error_type => var`形式で例外オブジェクトを取得でき、組み込み変数`$!`でも参照できます。発生箇所は`$@`に格納されます。

### begin

```ruby
begin
  do_something
rescue
  recover
ensure
  must_to_do
end
```

**文法:**

```
begin
  式..
[rescue [error_type,..] [=> evar] [then]
  式..]..
[else
  式..]
[ensure
  式..]
end
```

実行中に例外が発生した場合、rescue節で捕捉できます。発生した例外は`$!`で参照でき、指定されていれば変数evarにも格納されます:

```ruby
begin
  raise "error message"
rescue => evar
  p $!
  p evar
end
# => #<RuntimeError: error message>
#    #<RuntimeError: error message>
```

error_typeが省略されるとStandardErrorのサブクラスである全ての例外を捕捉します。

else節は本体で例外が発生しなかった場合に評価されます。

ensure節は常にbegin式終了直前に評価されます。

```ruby
begin
  式1
rescue
  式2
end
```

返り値は本体／rescue節／else節の最後に評価された文の値です。各節に文がなければnil、ensure節の値は無視されます。

クラス定義、モジュール定義、メソッド定義ではbeginなしでrescue、ensure節を定義できます。

### rescue修飾子

```ruby
open("nonexistent file") rescue STDERR.puts "Warning: #$!"
```

**文法:** `式1 rescue 式2`

式1で例外が発生したとき式2を評価します:

```ruby
begin
  式1
rescue
  式2
end
```

と同じです。StandardErrorサブクラスのみ捕捉できます。

```ruby
var = open("nonexistent file") rescue false
p var
# => false
```

優先順位により式全体を括弧で囲む必要があります:

```ruby
p((open("nonexistent file") rescue false))
=> false
```

---

## その他

### return

```ruby
return
return 12
return 1,2,3
```

**文法:**

```
return [式[,` 式 ... ]]
```

式の値をメソッドの戻り値として実行を終了します。複数式で配列を返します。式省略時はnil。

トップレベルでreturnするとプログラム終了、require/loadファイル内なら呼び出し元に返ります。

### BEGIN

```ruby
BEGIN {
  # ...
}
```

**文法:**

```
BEGIN '{' 文.. '}'
```

初期化ルーチンを登録します。複数あると指定順に実行され、コンパイル時に登録されます。

独立したスコープを導入しません。トップレベル以外では書けません:

```ruby
def foo
  BEGIN { p "begin" }
end
# => -e:2: syntax error, unexpected keyword_BEGIN

class Foo
  BEGIN { p "begin" }
end
# => -e:2: syntax error, unexpected keyword_BEGIN
```

### END

```ruby
END {
  # ...
}
```

**文法:**

```
END '{' 文.. '}'
```

後始末ルーチンを登録します。インタプリタ終了時に実行されます。

複数登録時は逆順で実行されます:

```ruby
END { p 1 }
END { p 2 }
END { p 3 }
# => 3
#    2
#    1
```

ループ内で実行しても最初の一回のみ有効です:

```ruby
5.times do |i|
  END { p i }
end
# => 0
```

メソッド定義式中では警告が出ます:

```ruby
def foo
  END { p "end" }
end
p foo
# => -:2: warning: END in method; use at_exit
#    nil
#    "end"
```

実行時に登録するため、実行されない場合があります:

```ruby
if false
  END { p "end" }
end
```

周囲とスコープを共有します。中で発生した例外はブロックを中断しますが、全ての後始末ルーチンが実行されます:

```ruby
END { p "FOO" }
END { raise "bar"; p "BAR" }
END { raise "baz"; p "BAZ" }
# => baz (RuntimeError)
#    bar (RuntimeError)
#    "FOO"
```

---

**ライセンス:** [Creative Commons License](https://creativecommons.org/licenses/by/3.0/)
