# クラス／メソッドの定義 (Ruby 3.4 リファレンスマニュアル)

## 目次

- [クラス定義](#クラス定義)
- [特異クラス定義](#特異クラス定義)
- [モジュール定義](#モジュール定義)
- [メソッド定義](#メソッド定義)
- [演算子式の定義](#演算子式の定義)
- [メソッド定義のネスト](#メソッド定義のネスト)
- [メソッドの評価](#メソッドの評価)
- [特異メソッド定義](#特異メソッド定義)
- [クラスメソッドの定義](#クラスメソッドの定義)
- [呼び出し制限](#呼び出し制限)
- [定義に関する操作](#定義に関する操作)

---

## クラス定義

クラスを定義する基本構文：

```ruby
class Foo < Super
  def test
    # ...
  end
  # ...
end
```

**文法：**

```
class 識別子 ['<' superclass]
  式..
end
```

### 説明

クラス名はアルファベットの大文字で始まる識別子です。rescue/ensure 節で例外処理ができます。

クラス定義は定数へのクラスの代入であり、RubyではクラスはClassクラスのインスタンスです。

既に定義されているクラスに対して再度定義することで定義の追加ができます。ただし、元のクラスと異なるスーパークラスを指定するとTypeErrorが発生します。

```ruby
class Foo < Array
  def foo
  end
end

# 定義を追加
class Foo
  def bar
  end
end

# エラー: スーパークラスが異なる
class Foo < String
end
# => superclass mismatch for class Foo (TypeError)
```

クラス定義式内ではselfがそのクラスであり、任意の式を記述できます。

### ネストしたクラス定義

クラスはネストして定義できます：

```ruby
class Foo
  class Bar
  end
end
```

クラス定義式は最後に評価した式の結果を返し、値を返さない場合はnilを返します。

---

## 特異クラス定義

```ruby
obj = Object.new
class << obj
  def test
    # ...
  end
  # ...
end
```

**文法：**

```
class '<<' expr
  式..
end
```

### 説明

特定のオブジェクトにメソッドやインスタンス変数を定義/追加します。定義したメソッドや定数は指定したオブジェクトに対してのみ有効です。

- `Object#clone`で生成したオブジェクトには引き継がれます
- `Object#dup`で生成したオブジェクトには引き継がれません

---

## モジュール定義

```ruby
module Foo
  def test
    # ...
  end
  # ...
end
```

**文法：**

```
module 識別子
  式..
end
```

### 説明

モジュール名はアルファベットの大文字で始まる識別子です。モジュールもオブジェクトであり、Moduleクラスのインスタンスです。

---

## メソッド定義

### 基本構文

```ruby
def fact(n)
  if n == 1 then
    1
  else
    n * fact(n-1)
  end
end
```

### 説明

メソッドを定義する場所によって：

- クラス/モジュール定義中：そのクラス/モジュールのメソッド
- トップレベル：どこからでも呼べるメソッド（「関数」のように機能）

### デフォルト引数

仮引数にデフォルト式を指定できます：

```ruby
def foo(x, y = 1)
  10 * x + y
end

p foo(1, 5)   #=> 15
p foo(3)      #=> 31
```

### 可変長引数（*args）

```ruby
def foo(x, *xs)
  puts "#{x} : #{xs.inspect}"
end

foo(1)         #=> 1 : []
foo(1, 2, 3)   #=> 1 : [2, 3]
```

### ブロック引数（&block）

```ruby
def foo(cnt, &block_arg)
  cnt.times { block_arg.call }
end

foo(3) { print "Ruby! " }  #=> Ruby! Ruby! Ruby!
```

### 引数の順序

仮引数はその種類ごとに以下の順序でのみ指定できます：

1. デフォルト式のない引数（複数可）
2. デフォルト式のある引数（複数可）
3. `*`を伴う引数（1つだけ）
4. デフォルト式のない引数（複数可）
5. キーワード引数（複数可）
6. `**`を伴う引数（1つだけ）
7. `&`を伴う引数（1つだけ）

---

## 演算子式の定義

再定義可能な演算子の実装はメソッドなので定義できます。

### 二項演算子

```ruby
def +(other); end    # obj + other
def -(other); end    # obj - other
```

### 単項プラス/マイナス

```ruby
def +@; end          # +obj
def -@; end          # -obj
```

### 要素代入

```ruby
def foo=(value); end              # obj.foo = value
def [](key); end                  # obj[key]
def []=(key, value); end          # obj[key] = value
```

---

## 特異メソッド定義

```ruby
def foo.test
  print "this is foo\n"
end
```

特異メソッドは特定のオブジェクトに固有のメソッドです。

---

## クラスメソッドの定義

Rubyでのクラスメソッドはクラスの特異メソッドです。

### 定義方法

**self を使う方式（推奨）：**

```ruby
class Hoge
  def self.baz
    'To infinity and beyond!'
  end
end
```

---

## 呼び出し制限

メソッドはpublic、private、protectedの3つの呼び出し制限を持ちます。

- **public**：制限なしに呼び出せます
- **private**：関数形式でしか呼び出せません
- **protected**：そのメソッドを持つオブジェクトがselfであるコンテキストでのみ呼び出せます

---

## 定義に関する操作

### alias

```ruby
alias foo bar
alias :foo :bar
```

メソッドあるいはグローバル変数に別名をつけます。

### undef

```ruby
undef bar
```

メソッドの定義を取り消します。

### defined?

```ruby
defined? print
defined? File.print
```

式が定義されていなければnilを返します。定義されていれば式の種別を表す文字列を返します。

---

## ライセンス

[![Creative Commons License](https://i.creativecommons.org/l/by/3.0/88x31.png)](https://creativecommons.org/licenses/by/3.0/)
