# リテラル (Ruby 3.4 リファレンスマニュアル)

## Overview

This is Ruby's documentation on literals—values written directly in code like `123` or `"hello world"`.

## 数値リテラル (Numeric Literals)

| Form | Description |
|------|-------------|
| `123`, `0d123` | Integer |
| `-123` | Signed integer |
| `123.45` | Floating point |
| `1.2e-3` | Floating point with exponent |
| `0xffff` | Hexadecimal |
| `0b1011` | Binary |
| `0377`, `0o377` | Octal |
| `42r`, `3.14r` | Rational numbers |
| `42i`, `3.14i` | Complex numbers |

Numeric literals support underscores for readability: `1_000_000_000`

## 文字列リテラル (String Literals)

Strings use double or single quotes:
- **Double quotes**: Support escape sequences and interpolation
- **Single quotes**: Minimal interpretation (`\\` and `\'` only)

```ruby
"this is a string\n"
'this is a string\n'
```

Adjacent string literals concatenate at compile time:
```ruby
p "foo" "bar"  # => "foobar"
```

### バックスラッシュ記法 (Escape Sequences)

| Sequence | Meaning |
|----------|---------|
| `\t` | Tab |
| `\n` | Newline |
| `\xnn` | Hex character |
| `\unnnn` | Unicode (4 digits) |
| `\u{nnnn}` | Unicode (1-6 digits) |
| `\\改行` | Line continuation |

### 式展開 (Interpolation)

Double-quoted strings support `#{expression}`:
```ruby
$ruby = "RUBY"
"my name is #{$ruby}"  # => "my name is RUBY"
```

Variables can omit braces: `#$variable`

## 文字リテラル (Character Literals)

```ruby
?a          # => "a"
?\u3042     # => "あ"
?\C-a       # Control character
```

## コマンド出力 (Command Output)

Backtick strings execute as shell commands:
```ruby
`date`
%x{ date }
```

Supports escape sequences and interpolation like double-quoted strings.

## ヒアドキュメント (Heredocs)

```ruby
print <<EOS
  the string
  next line
EOS
```

**Variants:**
- `<<-ID`: Indented terminator allowed
- `<<~ID`: Dedents content based on least-indented line

```ruby
str = <<~EOS
    indented
      more indented
EOS
```

**With quotes:**
- `<<"EOS"`: Interpolation enabled (default)
- `<<'EOS'`: No interpolation
- `` <<`EOS` ``: Execute as command

## 正規表現リテラル (Regex Literals)

```ruby
/^Ruby/
/Ruby/i    # case-insensitive
/foo.*bar/x  # extended (ignores whitespace)
```

**Options:**
- `i`: Case-insensitive
- `m`: Multiline (`.` matches newlines)
- `x`: Extended (whitespace ignored, comments allowed)
- `o`: Evaluate interpolation once

## 配列式 (Array Literals)

```ruby
[1, 2, 3]
%w(a b c)      # string array
%W(a #{x} c)   # string array with interpolation
```

## ハッシュ式 (Hash Literals)

```ruby
{ 1 => 2, 2 => 4 }
{ :a => "A", :b => "B" }
{ a: "A", b: "B" }  # symbol keys shorthand
```

## 範囲オブジェクト (Range Literals)

Numeric-literal ranges return identical objects on each evaluation; otherwise new objects.

## シンボル (Symbols)

```ruby
:class
:method!
:$gvar
:'foo-bar'
:"foo-bar"
%s{foo-bar}
```

Symbols are unique objects; identical expressions always return the same instance.

## %記法 (Percent Literals)

| Form | Type |
|------|------|
| `%!STRING!` or `%Q!STRING!` | Double-quote string |
| `%q!STRING!` | Single-quote string |
| `%x!STRING!` | Command output |
| `%r!STRING!` | Regex |
| `%w!STRING!` | String array |
| `%W!STRING!` | String array (interpolated) |
| `%i!STRING!` | Symbol array |
| `%I!STRING!` | Symbol array (interpolated) |

Delimiters can be any non-alphanumeric character; paired brackets automatically close:

```ruby
%(())  # => "()"
%w(foo\ bar baz)  # => ["foo bar", "baz"]
```
