# 演算子式 (Ruby 3.4 リファレンスマニュアル)

## 目次

- [代入](#代入)
- [自己代入](#自己代入)
- [多重代入](#多重代入)
- [範囲式](#範囲式)
- [条件式としての範囲式](#条件式としての範囲式)
- [and](#and)
- [or](#or)
- [not](#not)
- [条件演算子](#条件演算子)

## 概要

演算子の優先順位（高い順）：

```
::
[]
+(単項) ! ~
**
-(単項)
* / %
+ -
<< >>
&
| ^
> >= < <=
<=> == === != =~ !~
&&
||
.. ...
?:(条件演算子)
=(+=, -= ... )
not
and or
```

ほとんどの演算子はメソッド呼び出しですが、一部は言語に組み込みで再定義できません。

### 再定義可能な演算子

```
| ^ & <=> == === =~ > >= < <= << >>
+ - * / % ** ~ +@ -@ [] []= ` ! != !~
```

### 再定義不可能な演算子

```
= ?: .. ... not && and || or ::
```

---

## 代入

**構文：**

```
変数 '=' 式
定数 '=' 式
式'['expr..']' '=' 式
式'.'識別子 '=' 式
```

代入式の左辺は以下のいずれか：

### 変数への代入

```ruby
foo = bar
```

### 配列参照

```ruby
class C
  def initialize
    @ary = [0,1,2,3,4,5,6,7]
  end
  def [](i)
    @ary[i * 2]
  end
  def []=( i, v )
    @ary[i * 2] = v
  end
end

c = C.new
p c[3]      # c.[]( 3 )に変換され、結果は 6
p c[3] = 1  # c.[]=(3,1)に変換され、結果は 1
```

### 属性参照

```ruby
class C
  def foo
    @foo
  end
  def foo=( v )
    @foo = v
  end
end

c = C.new
c.foo = 5   # c.foo=( 5 )のように変換される
p c.foo     # => 5
```

または `Module#attr` を使用：

```ruby
class C
  attr :foo, true
end

c = C.new
c.foo = 5
p c.foo     # => 5
```

---

## 自己代入

**構文：**

```
式1 op= 式2
```

`op` は以下のいずれか：

```
+, -, *, /, %, **, &, |, ^, <<, >>, &&, ||
```

**評価方式：**

通常の場合：
```
式1 = 式1 op 式2
```

`&&`, `||` の場合：
```
式1 op (式1 = 式2)
```

**例：**

```ruby
foo += 12       # foo = foo + 12
a ||= 1         # a が偽か未定義ならば1を代入
```

属性参照での違い：

```ruby
obj.foo ||= true
# 以下と同じ：
obj.foo || (obj.foo = true)
# 以下ではない：
obj.foo = obj.foo || true
```

---

## 多重代入

**構文：**

```
式 [',' [式 ',' ... ] ['*' [式]]] = 式 [, 式 ... ]['*' 式]
'*' [式] = 式 [, 式 ... ]['*' 式]
```

**動作：**

複数の式または配列から同時に代入。右辺が1つの場合は配列に変換される。

**例：**

```ruby
foo, bar = [1, 2]       # foo = 1; bar = 2
foo, bar = 1, 2         # foo = 1; bar = 2
foo, bar = 1            # foo = 1; bar = nil
foo, bar, baz = 1, 2    # foo = 1; bar = 2; baz = nil
foo, bar = 1, 2, 3      # foo = 1; bar = 2
foo = 1, 2, 3           # foo = [1, 2, 3]
*foo = 1, 2, 3          # foo = [1, 2, 3]
foo,*bar = 1, 2, 3      # foo = 1; bar = [2, 3]
```

**ネストした配列：**

```ruby
(foo, bar), baz = [1, 2], 3       # foo = 1; bar = 2; baz = 3
```

**特殊な形式：**

```ruby
class C
  def foo=( v )
    @foo = v
  end
  def []=(i,v)
    @bar = ["a", "b", "c"]
    @bar[i] = v
  end
end

obj = C.new
obj.foo, obj[2] = 1, 2     # @foo = 1; @bar = ["a", "b", 2]
```

**末尾がコンマの場合：**

```ruby
foo,* = 1, 2, 3      # foo = 1
foo, = 1, 2, 3       # foo = 1
* = 1, 2, 3
```

**メソッド定義での使用：**

```ruby
def foo(*) end
foo(1,2,3)
```

---

## 範囲式

**構文：**

```
式1 '..' 式2
式1 '...' 式2
式1 '..'
式1 '...'
```

**動作：**

`Range` オブジェクトを返す。`...` は終端を含みません。

**例：**

```ruby
1 .. 20
/first/ ... /second/
(1..)
```

終端なし Range の例：
```ruby
(1..)        # (1..nil)の構文糖
```

注意：`when 1..` は行継続とみなされるため、括弧が必要な場合があります。

---

## 条件式としての範囲式

フリップフロップ (flip-flop) とも呼ばれます。

### 「..」の場合：

1. 初期状態では式1だけ評価し、真になるまで `false` を返す
2. 式1が真になると `true` を返す。式2が真なら初期状態に戻る
3. その後、式2だけ評価し、真になるまで `true` を返す
4. 式2が真になると `true` を返したあと、初期状態に戻る

### 「...」の場合：

1. 初期状態では式1だけ評価し、真になるまで `false` を返す
2. 式1が真になると `true` を返す
3. その後、式2だけ評価し、真になるまで `true` を返す
4. 式2が真になると `true` を返したあと、初期状態に戻る

**例：**

```ruby
5.times{|n|
  if (n==2)..(n==3)
    p n
  end
}
#=> 2
#=> 3

5.times{|n|
  if (n==2)...(n==3)
    p n
  end
}
#=> 2
#=> 3

5.times{|n|
  if (n==2)..(n==2)
    p n
  end
}
#=> 2

5.times{|n|
  if (n==2)...(n==2)
    p n
  end
}
#=> 2
#=> 3
#=> 4
```

---

## and

**構文：**

```
式 '&&' 式
式 'and' 式
```

**動作：**

左辺が偽なら `nil` または `false` を返す。真なら右辺を評価して返す。

```ruby
p(nil && false) # => nil
p(false && nil) # => false
p(1 && 2) # => 2
```

メソッド引数での使用：

```ruby
p(true && false)    #=> false
p((true and false)) #=> false
```

---

## or

**構文：**

```
式 '||' 式
式 or 式
```

**動作：**

左辺が真なら返す。偽なら右辺を評価して返す。

```ruby
p(1 || 2) # => 1
p(nil || false) # => false
p(false || nil) # => nil
```

メソッド引数での使用：

```ruby
p(false || true)    #=> true
p((false or true)) #=> true
```

---

## not

**構文：**

```
'!' 式
not 式
```

**動作：**

真なら偽を、偽なら真を返す。

```ruby
! me
not me
```

**関連演算子：**

```
式 '!=' 式    # !(式 == 式)と同じ
式 '!~' 式    # !(式 =~ 式)と同じ
```

メソッド引数での使用：

```ruby
p(! false)      #=> true
p((not false))  #=> true
```

---

## 条件演算子

**構文：**

```
式1 ? 式2 : 式3
```

**動作：**

式1の結果によって式2または式3を返す。以下と同じ：

```ruby
if 式1 then 式2 else 式3 end
```

**例：**

```ruby
obj == 1 ? foo : bar
```

---

**ライセンス：** [Creative Commons License](https://creativecommons.org/licenses/by/3.0/) [![Creative Commons License](https://i.creativecommons.org/l/by/3.0/88x31.png)](https://creativecommons.org/licenses/by/3.0/)

[フィードバックを送る](https://github.com/rurema/doctree/issues/new) / [このマニュアルを編集する](https://github.com/rurema/doctree/edit/master/refm/doc/spec/operator.rd)
