# パターンマッチ (Ruby 3.4 リファレンスマニュアル)

## 目次

- [パターン](#パターン)
- [変数の束縛](#変数の束縛)
- [変数のピン留め](#変数のピン留め)
- [非プリミティブなオブジェクトのマッチ](#非プリミティブなオブジェクトのマッチ)
- [ガード節](#ガード節)

## 概要

パターンマッチは構造化された値の構造をチェックし、マッチした部分をローカル変数に代入する機能です。Ruby では `case/in` 式を使って実装されています。

```ruby
case <expression>
in <pattern1>
  ...
in <pattern2>
  ...
else
  ...
end
```

`case/in` 式の主な特徴：
- `in` 節と `when` 節は混在できません
- マッチしなかった場合、`NoMatchingPatternError` 例外が発生します（`else` 節がない場合）

### 単一式での使用

```ruby
<expression> => <pattern>
<expression> in <pattern>
```

## パターン

利用可能なパターンの種類：

- **Value パターン**: すべての Ruby オブジェクト（`===` 演算子でマッチ）
- **Array パターン**: `[<subpattern>, <subpattern>, ...]`
- **Find パターン**: `[*variable, <subpattern>, ..., *variable]`
- **Hash パターン**: `{key: <subpattern>, key: <subpattern>, ...}`
- **Alternative パターン**: `|` でパターンを組み合わせ
- **As パターン**: `<pattern> => variable`
- **Variable パターン**: `variable`

### Array パターンの特性

配列では**すべての要素**がマッチする必要があります：

```ruby
case [1, 2, 3]
in [Integer, Integer]
  "matched"
else
  "not matched"
end
#=> "not matched"
```

### Hash パターンの特性

ハッシュは一部のキーのみ指定した場合でもマッチします：

```ruby
case {a: 1, b: 2, c: 3}
in {a: Integer}
  "matched"
else
  "not matched"
end
#=> "matched"
```

### Find パターン

複数要素の中から特定パターンを検出：

```ruby
case ["a", 1, "b", "c", 2]
in [*, String, String, *]
  "matched"
end
```

## 変数の束縛

マッチした値をローカル変数に代入します。

```ruby
case [1, 2]
in Integer => a, Integer
  "matched: #{a}"
end
#=> "matched: 1"
```

Hash パターンでのキー指定による自動束縛：

```ruby
case {a: 1, b: 2, c: 3}
in a:
  "matched: #{a}"
end
#=> "matched: 1"
```

## 変数のピン留め

既存のローカル変数を値として使用する場合、ピン演算子 `^` を使用します：

```ruby
expectation = 18
case [1, 2]
in ^expectation, *rest
  "matched"
else
  "not matched"
end
#=> "not matched"
```

## 非プリミティブなオブジェクトのマッチ

`deconstruct` メソッド（Array/Find パターン用）と `deconstruct_keys` メソッド（Hash パターン用）を定義したオブジェクトもマッチ対象です。

```ruby
class Point
  def initialize(x, y)
    @x, @y = x, y
  end

  def deconstruct
    [@x, @y]
  end

  def deconstruct_keys(keys)
    {x: @x, y: @y}
  end
end

case Point.new(1, -2)
in px, Integer
  "matched: #{px}"
end
#=> "matched: 1"
```

## ガード節

`if` または `unless` を使ってマッチ後の追加条件式を指定できます：

```ruby
case [1, 2]
in a, b if b == a*2
  "matched"
else
  "not matched"
end
#=> "matched"
```

---

**ライセンス**: [Creative Commons Attribution 3.0](https://creativecommons.org/licenses/by/3.0/)
