# 変数と定数 (Ruby 3.4 リファレンスマニュアル)

## 目次

- [ローカル変数](#ローカル変数)
- [インスタンス変数](#インスタンス変数)
- [クラス変数](#クラス変数)
- [クラス変数のスコープ](#クラス変数のスコープ)
- [グローバル変数](#グローバル変数)
- [擬似変数](#擬似変数)
- [定数](#定数)
- [定数参照の優先順位](#定数参照の優先順位)

---

## 概要

Ruby では、変数と定数の種別は「識別子の最初の文字」で区別されます。一部の組み込み変数は「$」に続く記号や数字で構成されます。変数名の長さに制限はなく、2文字目以降は英数字またはアンダースコアが使用できます。

## ローカル変数

**例:** `foobar`

小文字またはアンダースコアで始まる識別子はローカル変数またはメソッド呼び出しになります。

### スコープ

- 宣言位置からブロック、メソッド定義、またはクラス/モジュール定義の終わりまで有効
- 宣言は実行されなくても有効

```ruby
2.times {
  p defined?(v)    # nil
  v = 1
  p v              # 1
}
# => nil
#    1
#    nil
#    1
```

```ruby
v = 1 if false
p defined?(v)  # => "local-variable"
p v            # => nil
```

## インスタンス変数

**例:** `@foobar`

「@」で始まる変数はインスタンス変数です。特定のオブジェクトに属し、そのクラスまたはサブクラスのメソッドから参照できます。初期化されていない場合の値は `nil` です。

## クラス変数

**例:**
```ruby
class Foo
  @@foo = 1
  def bar
    puts @@foo
  end
end
```

「@@」で始まる変数はクラス変数です。クラス定義内で定義され、クラスの特異メソッドやインスタンスメソッドから参照・代入できます。

### 定数との違い

- 再代入可能（定数は警告を出す）
- クラス外から直接参照できない（継承クラスからは可能）

### クラス自身のインスタンス変数との違い

- サブクラスから参照・代入が可能
- インスタンスメソッドから参照・代入が可能

### 継承時の共有

```ruby
class Foo
  @@foo = 1
end

class Bar < Foo
  p @@foo += 1      # => 2
end

class Baz < Bar
  p @@foo += 1      # => 3
end
```

### モジュール変数の共有

```ruby
module Foo
  @@foo = 1
end

class Bar
  include Foo
  p @@foo += 1      # => 2
end

class Baz
  include Foo
  p @@foo += 1      # => 3
end
```

### スコープ外での参照エラー

親クラスで同名のクラス変数を追加すると、`RuntimeError` が発生する場合があります。

### クラス変数のスコープ

最も内側の非特異クラス/モジュール式のボディがスコープです。

```ruby
class Foo
  @@a = :a

  class << Foo
    p @@a           #=> :a
  end

  def Foo.a1
    p @@a
  end
end

Foo.a1              #=> :a

def Foo.a2
  p @@a
end
Foo.a2              #=> NameError

class << Foo
  p @@a             #=> NameError
end
```

## グローバル変数

**例:** `$foobar`, `$/`

「$」で始まる変数はグローバル変数です。プログラムのどこからでも参照できます。宣言は不要で、初期化されていない場合の値は `nil` です。

### 組み込み変数

Ruby 処理系によって特殊な意味を与えられた変数を組み込み変数と呼びます。詳細は Kernel の特殊変数を参照してください。

### 特殊変数とオプション変数

**特殊変数:** `$'`, `$&`, `$1`, `$2` など（「$」+特殊文字または数字）

**オプション変数:** `$-F`, `$-I` など（Ruby の起動オプションに対応）

### スコープの種類

| スコープ | 説明 |
|---------|------|
| グローバルスコープ | 通常のグローバル変数と同じ |
| ローカルスコープ | 通常のローカル変数と同じ |
| スレッドローカルスコープ | スレッドごとの独立した値 |

一部の変数は読み取り専用で、代入しようとすると例外が発生します。

## 擬似変数

通常の変数ではない特殊な変数です。

| 変数 | 説明 |
|------|------|
| `self` | 現在のメソッドの実行主体 |
| `nil` | `NilClass` の唯一のインスタンス（真理値は偽） |
| `true` | `TrueClass` の唯一のインスタンス（真理値は真） |
| `false` | `FalseClass` の唯一のインスタンス（真理値は偽） |
| `__FILE__` | 現在のソースファイル名 |
| `__LINE__` | 現在のソースファイル中の行番号 |
| `__ENCODING__` | 現在のソースファイルのスクリプトエンコーディング |

> 擬似変数の値は変更できません。代入すると構文エラーになります。

## 定数

**例:** `FOOBAR`, `ＦＯＯＢＡＲ`

アルファベット大文字 `[A-Z]` で始まる識別子は定数です。Unicode 対応の場合、Unicode の大文字やタイトルケース文字からも定義できます。

### 特性

- 代入によって定義・初期化
- メソッド内では定義不可
- 再代入で警告メッセージが出力される
- 定義されていない定数にアクセスすると `NameError` 発生

### 参照可能な範囲

定義されたクラス/モジュール内（メソッド本体やネストしたクラス/モジュール含む）、継承クラス、インクルードしているクラス/モジュール。トップレベルで定義された定数は `Object` に所属します。

```ruby
class Foo
  FOO = 'FOO'
end

class Bar < Foo
  BAR = 'BAR'

  p FOO             # => "FOO"

  class Baz
    p BAR           # => "BAR"
  end
end
```

### :: 演算子

外部から定数を参照する場合、`::` 演算子を使用します。

```ruby
module M
  I = 35
  class C
  end
end

p M::I              #=> 35
p M::C              #=> M::C
p ::M               #=> M
M::NewConst = 777   # => 777
```

## 定数参照の優先順位

### :: で始まる定数参照

`Object` クラスを起点にスーパークラスを順番に探索します。

```ruby
X = 1

module M
  X = 2
  p ::X             # => 1
end
```

### :: を含まない定数参照

ソースコード上のクラスネストを外側に向かって探索し、最初に見つかった定数を参照。ネストの最外側まで探索しても見つからない場合、起点クラスのスーパークラスを順番に探索します。

```ruby
X = "Object"

class C
  X = "C"
end

module M1
  X = "M1"
  module M2
    class C3 < C
      # C3 -> M2 -> M1 -> C -> Object の順で探索
      p X # => "M1"
    end
  end
end
```

**注:** スコープの場所によっては外側のネストが探索されない場合があります。

```ruby
X = "Object"

class C
  X = "C"
end

module M1
  X = "M1"
end

module M1::M2
  class C3 < C
    # C3 -> M2 -> C -> Object の順で探索
    # M1 は探索されない
    p X # => "C"
  end
end
```

ネストの一覧は `Class.nesting` または `Module.nesting` で取得できます。

```ruby
module M1
  module M2
    p Module.nesting # => [M1::M2, M1]
  end
end

module M1::M2
  p Module.nesting # => [M1::M2]
end
```

### 他の値から :: で繋げられた定数参照

「::」前のクラスを起点にスーパークラスを順番に探索。ただし `Object` クラスとそのスーパークラスは探索対象外です。

```ruby
X = "Object"
Y = "Object"

class C
  X = "C"
end

class D < C
end

p D::X              # => "C"
p D::Y              # => uninitialized constant D::Y (NameError)
```

クラスのネストの外側は探索対象外です。

```ruby
module M
  X = "M"
  class C
  end
  p C::X            # => uninitialized constant M::C::X (NameError)
end
```

---

**ライセンス:** [Creative Commons Attribution 3.0](https://creativecommons.org/licenses/by/3.0/)
